import streamlit as st
import pandas as pd
import numpy as np
import matplotlib.pyplot as plt


def show():
    st.markdown("### 📊 数据分析面板")
    st.info("📌 您可以上传 CSV 文件，并使用筛选器和时间范围自定义视图")

    # ========================
    # 数据加载与上传
    # ========================
    uploaded_file = st.file_uploader("📁 上传 CSV 文件", type="csv")
    # CSV 文件格式提示
    st.code('''
    请上传的 CSV 文件包含以下列，示例如下：
    日期,销售额,访问量,转化率
    2023-01-01,2345,560,0.32
    2023-01-02,3456,780,0.41

    📌 注意事项：
    - 确保第一列为**日期**，格式为 `YYYY-MM-DD`
    - 其他列应为数值类型（整数或浮点数）
    ''', language='python')

    if uploaded_file:
        data = pd.read_csv(uploaded_file)

        # 自动识别日期列并尝试转换为 datetime 类型
        for col in data.columns:
            try:
                data[col] = pd.to_datetime(data[col])
            except Exception:
                pass

        # 确保包含必要的数值列
        required_columns = ['销售额', '访问量', '转化率', '日期']
        missing_cols = [col for col in required_columns if col not in data.columns]
        if missing_cols:
            st.warning(f"⚠️ 缺少必要字段：{', '.join(missing_cols)}，请检查上传的 CSV 格式是否正确。")
            return

    else:
        # 使用示例数据
        st.warning("🔔 当前显示为示例数据，请上传您自己的 CSV 文件以继续分析。")
        np.random.seed(42)
        data = pd.DataFrame({
            '日期': pd.date_range(start='2023-01-01', periods=30),
            '销售额': np.random.randint(1000, 5000, 30),
            '访问量': np.random.randint(200, 1000, 30),
            '转化率': np.random.uniform(0.1, 0.5, 30)
        })

    # ========================
    # 时间范围选择器
    # ========================
    min_date = data['日期'].min().to_pydatetime().date()
    max_date = data['日期'].max().to_pydatetime().date()

    date_range = st.slider(
        "📅 选择时间范围",
        min_value=min_date,
        max_value=max_date,
        value=(min_date, max_date)
    )

    # ========================
    # 数据筛选器
    # ========================
    with st.expander("🔍 高级筛选器"):
        sales_range = st.slider("💰 销售额范围", int(data['销售额'].min()), int(data['销售额'].max()),
                                (int(data['销售额'].min()), int(data['销售额'].max())))
        visit_range = st.slider("👥 访问量范围", int(data['访问量'].min()), int(data['访问量'].max()),
                                (int(data['访问量'].min()), int(data['访问量'].max())))

    # ========================
    # 应用筛选条件
    # ========================
    start_date = pd.Timestamp(date_range[0])
    end_date = pd.Timestamp(date_range[1])

    filtered_data = data[
        (data['日期'] >= start_date) &
        (data['日期'] <= end_date) &
        (data['销售额'] >= sales_range[0]) &
        (data['销售额'] <= sales_range[1]) &
        (data['访问量'] >= visit_range[0]) &
        (data['访问量'] <= visit_range[1])
    ]

    if filtered_data.empty:
        st.warning("🚫 当前筛选条件下无数据，请调整筛选参数。")
        return

    # ========================
    # 展示数据表格
    # ========================
    st.markdown("#### 📄 数据预览（已筛选）")
    st.dataframe(filtered_data, use_container_width=True)

    st.divider()

    # ========================
    # 图表展示区域
    # ========================
    st.markdown("#### 📈 可视化分析")
    tab1, tab2, tab3 = st.tabs(["📈 折线图", "📊 柱状图", "📉 散点图"])

    with tab1:
        fig, ax = plt.subplots(figsize=(10, 4))
        ax.plot(filtered_data['日期'], filtered_data['销售额'],
                marker='o', color='#1f77b4', linestyle='-')
        ax.set_title('销售额趋势', fontsize=14)
        ax.grid(True, linestyle='--', alpha=0.6)
        ax.set_xlabel('日期')
        ax.set_ylabel('销售额')
        plt.xticks(rotation=45)
        st.pyplot(fig)

    with tab2:
        fig, ax = plt.subplots(figsize=(10, 4))
        ax.bar(filtered_data['日期'], filtered_data['访问量'], color='#2ca02c')
        ax.set_title('每日访问量', fontsize=14)
        ax.grid(True, axis='y', linestyle='--', alpha=0.6)
        ax.set_xlabel('日期')
        ax.set_ylabel('访问量')
        plt.xticks(rotation=45)
        st.pyplot(fig)

    with tab3:
        fig, ax = plt.subplots(figsize=(8, 6))
        scatter = ax.scatter(
            filtered_data['访问量'],
            filtered_data['销售额'],
            c=filtered_data['转化率'],
            cmap='viridis',
            s=filtered_data['转化率'] * 100,
            alpha=0.7
        )
        ax.set_title('访问量 vs 销售额（气泡大小表示转化率）', fontsize=14)
        ax.set_xlabel('访问量')
        ax.set_ylabel('销售额')
        ax.grid(True, linestyle='--', alpha=0.6)
        fig.colorbar(scatter, ax=ax, label='转化率')
        st.pyplot(fig)

