import os
import json
import sqlite3

import streamlit as st
from streamlit_lottie import st_lottie
from streamlit_option_menu import option_menu


# 全局变量
DB_FILE = "./users.db"


# 初始化数据库连接
def get_db_connection():
    conn = sqlite3.connect(DB_FILE)
    return conn


# 初始化登录状态
def init_session():
    if 'logged_in' not in st.session_state:
        st.session_state.logged_in = False
    if 'username' not in st.session_state:
        st.session_state.username = ""
    if 'page' not in st.session_state:
        st.session_state.page = "login"


# 加载本地 Lottie 动画
def load_lottie_local(file_path):
    if os.path.exists(file_path):
        with open(file_path, 'r') as f:
            return json.load(f)
    return None


# 验证登录
def verify_login(username, password):
    conn = get_db_connection()
    c = conn.cursor()
    c.execute("SELECT password FROM users WHERE username = ?", (username,))
    result = c.fetchone()
    conn.close()
    return result and result[0] == password


# 登录页面
def login_page():
    st.markdown("<h1 style='text-align: center;'>🔐 欢迎登录智能数据分析平台</h1>", unsafe_allow_html=True)
    st.markdown("---")

    col1, col2 = st.columns([2, 1])

    with col1:
        st.markdown("### 🎯 请输入账号信息")
        username = st.text_input("账号", placeholder="请输入账号名", key="login_username")
        password = st.text_input("密码", type="password", placeholder="请输入密码", key="login_password")
        
        col_btn1, col_btn2 = st.columns(2)

        with col_btn1:
            if st.button("🚀 登录", use_container_width=True):
                if username and password:
                    if verify_login(username, password):
                        st.session_state.logged_in = True
                        st.session_state.username = username
                        st.session_state.page = "home"
                        st.rerun()
                    else:
                        st.error("❌ 账号或密码错误")
                else:
                    st.warning("⚠️ 请输入账号和密码")
        with col_btn2:
            if st.button("🆕 注册", use_container_width=True):
                st.session_state.page = "register"
                st.rerun()
    with col2:
        lottie_file = os.path.join("lottie", "login.json")
        lottie_login = load_lottie_local(lottie_file)
        if lottie_login:
            st_lottie(lottie_login, height=300)
        else:
            st.warning("登录动画文件未找到")


# 注册页面
def register_page():
    st.markdown("<h1 style='text-align: center;'>🆕 注册新账号</h1>", unsafe_allow_html=True)
    st.markdown("---")

    col1, col2 = st.columns([2, 1])

    with col1:
        st.markdown("### 📝 创建新账号")
        new_username = st.text_input("账号", placeholder="请输入新账号名", key="reg_username")
        new_password = st.text_input("密码", type="password", placeholder="请输入密码", key="reg_password")
        confirm_password = st.text_input("确认密码", type="password", placeholder="请再次输入密码", key="reg_confirm_password")

        col_btn1, col_btn2 = st.columns(2)

        with col_btn1:
            if st.button("✅ 提交注册", use_container_width=True):
                if new_username and new_password and confirm_password:
                    if new_password != confirm_password:
                        st.error("❌ 两次输入的密码不一致")
                    else:
                        conn = get_db_connection()
                        c = conn.cursor()
                        try:
                            c.execute("INSERT INTO users (username, password) VALUES (?, ?)", 
                                     (new_username, new_password))
                            conn.commit()
                            st.success("🎉 注册成功！请登录")
                            st.session_state.page = "login"
                            st.rerun()
                        except sqlite3.IntegrityError:
                            st.error("❌ 该账号已被注册")
                        finally:
                            conn.close()
                else:
                    st.warning("⚠️ 所有字段都必须填写")
        with col_btn2:
            if st.button("🔙 返回登录", use_container_width=True):
                st.session_state.page = "login"
                st.rerun()
    with col2:
        lottie_file = os.path.join("lottie", "register.json")
        lottie_register = load_lottie_local(lottie_file)
        if lottie_register:
            st_lottie(lottie_register, height=300)
        else:
            st.warning("注册动画文件未找到")


# 登录后的页面内容
def home_page():
    st.markdown(f"<h1 style='text-align: center;'>🏠 欢迎回来，{st.session_state.username}！</h1>", unsafe_allow_html=True)
    st.markdown("---")

    # 侧边栏导航菜单
    with st.sidebar:
        selected = option_menu(
            menu_title="导航",
            options=["我的主页", "数据分析", "关于"],
            icons=["house", "bar-chart", "info-circle"],
            menu_icon="menu-app",
            default_index=0,
            styles={
                "container": {"padding": "5px", "background-color": "#f0f2f6"},
                "icon": {"color": "orange", "font-size": "18px"},
                "nav-link": {"font-size": "16px", "text-align": "left", "margin": "3px"},
                "nav-link-selected": {"background-color": "#4CAF50"}
            }
        )

        if st.button("🚪 登出", use_container_width=True):
            st.session_state.logged_in = False
            st.session_state.username = ""
            st.session_state.page = "login"
            st.rerun()

    # 选择不同子页面并显示
    if selected == "我的主页":
        from subpages import home
        home.show()
    elif selected == "数据分析":
        from subpages import data_analysis
        data_analysis.show()
    elif selected == "关于":
        from subpages import about
        about.show()


# 主程序
def main():

    # 页面配置
    st.set_page_config(
        page_title="智析平台",
        page_icon="🔐",
        layout="wide",
        initial_sidebar_state="collapsed"
    )

    # 自定义CSS样式
    st.markdown("""
    <style>
        .stButton>button {
            width: 100%;
            transition: all 0.3s ease;
        }
        .stButton>button:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
        .header-text {
            text-align: center;
            font-size: 2.5rem !important;
            margin-bottom: 1rem;
            background: linear-gradient(135deg, #6a11cb 0%, #2575fc 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        .card {
            padding: 2rem;
            border-radius: 15px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            background: white;
            margin-bottom: 2rem;
        }
        .divider {
            height: 1px;
            background: linear-gradient(to right, transparent, #6a11cb, transparent);
            margin: 1.5rem 0;
        }
    </style>
    """, unsafe_allow_html=True)

    # 初始化登录状态
    init_session()

    # 页面路由（记录登录状态）
    if not st.session_state.logged_in:
        if st.session_state.page == "register":
            register_page()
        else:
            login_page()
    else:
        home_page()


if __name__ == "__main__":
    main()

